# -----------------------------------------------------------------------------
# uwoceanfun.py - Helper functions used by David Finlayson's ArcGIS scripts
#
# Version: 1.0
# August 26, 2004
# -----------------------------------------------------------------------------
# Copyright 2004 David Finlayson, Some Rights Reserved
#
# This script is licensed for use under a Creative Commons License.
# Details about the specific rights granted to you under this license can
# be found at the following URL:
#
# http://creativecommons.org/licenses/by-nc-sa/2.0/
#
# -----------------------------------------------------------------------------
#
# CONTACT INFO:
#
# David Finlayson
# dfinlays@u.washington.edu
# http://students.washington.edu/dfinlays
#
# Marine Geology & Geophysics
# School of Oceanography
# Box 357940
# University of Washington
# Seattle, WA  98195-7940
# USA
#
# REQUIREMENTS:
#
#  - ESRI ArcGIS 9.X (ArcInfo License)
#  - Spatial Analyst Extension
# -----------------------------------------------------------------------------

import os

class UWGeoprocessor:
    """ Contains functions that simplify calls to the real geoprocessor """
    def __init__(self, GP):
        self.GP = GP          # geoprocessing object
        self.assigned = 0     # id of assigned temp filenames
                    
    def clipToGrid(self, inGrid, outGrid, clipGrid):
        """ Clips a grid to the size of another (smaller) grid """
        # Get the clip raster extent
        desc = self.GP.Describe(clipGrid)
        extent = desc.Extent
        self.GP.Clip_management(inGrid, extent, outGrid, inGrid)

    def deleteFloat(self, inFloat):
        """ permanently deletes a Float binary file if it exists on disk """
##        GP.AddMessage("deleting: " + str(inFloat))
        if os.path.exists(inFloat):
            os.remove(inFloat)
            os.remove(inFloat[:-3] + "hdr")
##            
##            GP.AddMessage("deleting: " + str(inFloat[:-3] + "hdr"))
##            
##            os.remove(inFloat + ".xml")
####            os.remove(inFloat[:-3] + "prj")

    def deleteGrid(self, inGrid):
        """ permanently deletes a grid if it exists on disk """
        
        if self.GP.Exists(inGrid):
            self.GP.Delete(inGrid)
            

    def findGridCenter(self, inGrid):
        """ returns the center of a grid """
        #Get the raster extent (Xmin, Ymin, Xmax, Ymax)
        desc = self.GP.Describe(inGrid)
        extent = desc.Extent
        extent = [float(i) for i in extent.split()]

        # Find the center of the GRID
        x = extent[0] + (extent[2] - extent[0])/2.0
        y = extent[1] + (extent[3] - extent[1])/2.0
        center = "%14f %14f" % (x, y)
        center = center.strip()
        return(center)

    def floatToRaster(self, inFloat, outRaster):
        """ converts a floating point binary file to a GRID """
        self.GP.FloatToRaster_conversion(inFloat, outRaster)

    def rasterToFloat(self, inRaster, outFloat):
        """ converts a GRID to a floating point binary file """
        self.GP.RasterToFloat_conversion(inRaster, outFloat)

    def rotateGrid(self, inGrid, outGrid, Angle, pivot=None):
        """ rotates a grid about a pivot point (default pivot = center of grid)"""
        # If no pivot point is given, use the center of the grid
        if pivot == None:
            pivot = self.findGridCenter(inGrid)        
        self.GP.Rotate_management(inGrid, outGrid, str(Angle % 360.0), pivot, "NEAREST")            
        return(pivot)

    def tempFloat(self):
        """ returns a unique temporary floating point binary file name """
        while True:
            self.assigned = self.assigned + 1
            dataname = "temp" + str(self.assigned).zfill(3) + ".flt"
            datapath = os.path.join(self.GP.Scratchworkspace, dataname)

            headername = "temp" + str(self.assigned).zfill(3) + ".hdr"
            headerpath = os.path.join(self.GP.Scratchworkspace, headername)

            if not os.path.exists(datapath) or not os.path.exists(headerpath):
                break
        return(datapath)

    def tempGrid(self):
        """ returns a unique temporary grid name """
        while True:
            self.assigned = self.assigned + 1
            filename = "temp" + str(self.assigned).zfill(3) + ".tif" # jjr add
            fullpath = os.path.join(self.GP.Scratchworkspace, filename)

            if not self.GP.Exists(fullpath):
                break
        self.assigned = self.assigned + 1
        return(fullpath)
