# -----------------------------------------------------------------------------
# fetch_jjr.py - Calculate fetch using a text file of multiple angles.
# 04-09-25 - Date of script update
#
# REQUIREMENTS:
#
#  - ESRI ArcGIS Pro (Beta tested using 3.3.1)
#  - Python (Beta tested using version 3.11.8)
#  - Spatial Analyst Extension
#  - uwoceanfun.py (a library of helper functions developed by David Finlayson, USGS-
#    Pacific Science Center)
#
# Wind fetch is defined as the unobstructed distance that wind can travel over water
# in a constant direction.  Fetch is an important characteristic of open water because
# longer fetch can result in larger wind-generated waves. The larger waves, in turn,
# can increase shoreline erosion and sediment resuspension.  Wind fetches
# in this model were calculated using scripts designed by David Finlayson,
# U. S. Geological Survey, Pacific Science Center, while he was a Ph.D. student at the
# University of Washington (see information below).  This method calculates effective
# fetch using the recommended procedure of the Shore Protection Manual (USACOE 1984).
# This method spreads nine radials around the desired wind direction at 3-degree
# increments.  The resultant wind fetch is the arithmetic mean of these nine radial
# measurements.
#
# This script was developed based upon the original wind fetch script developed by
# David Finlayson.  This tool calculates wind fetch lengths
# for multiple wind directions based upon an input land/water raster.
#
# The wind fetch scripts that the model operates from were developed by Finlayson
# using the Python scripting language and were originally designed to run on the
# ArcGIS 9.0 (Environmental Systems Research Institute ([ESRI] Redlands, California)
# Geographic Information System (GIS) platform.  However, these scripts needed to
# be updated in order to operate using the most current ArcGIS revision, 10.0.  The
# model was also modified to more efficiently meet the needs of USACE planning
# personnel.  This modification gives the model the ability to calculate wind fetch
# for multiple wind directions based upon a text file listing individual compass
# directions.  
#
# Each of the individual directional wind fetch outputs are saved to a specified
# output workspace and named according to their respective calculated wind direction.
# When the model is initiated, it defaults to "SPM" as the desired calculation method.
# The other option is "Single" which calculates wind fetch on a single radial instead
# of the nine radials recommended in the shoreline protection manual.
#
#
# <Land_Raster>                 - Full path to a land raster. Values > 0.0 indicate land,
#                               Values = 0.0 indicate water.
#
#
# <Wind_Direction_Weighting_Percent_List>
#                               - A text file (.txt) containing values of wind directions and percent weightings 
#                               (optional).  These values must be between 0.0 and 360.0 indicating the direction
#                               FROM which the wind is blowing. Fetch length is calculated along this bearing.
#                               The percent weighting values must be between 0.0 and 100.0 and the total percent  
#                               for all wind directions must be equal to 100.0. Format for these inputs
#                               should be in the form wind direction value then a comma, then the percent weighting.
#
#
# <SINGLE|SPM|SPM_Restricted>   - One of the following strings: "SPM" or "SINGLE".
#
#                               "SPM" uses the recommended procedure of the
#                               Shore Protection Manual for calculating
#                               fetch length. This method spreads 9 radials
#                               about the WIND_DIRECTION at 3-degree
#                               increments. The reported fetch length is the
#                               arithmetic mean of the length of the 9
#                               radials between the cell center and land.
#
#                               "SPM-Restricted"  This method spreads 5 radials about
#                               the WIND_DIRECTION at 3-degree increments. The reported
#                               fetch length is the arithmetic mean of the length of the
#                               5 radials between the cell center and land.
#
#                               "SINGLE" calculates fetch length along a
#                               single radial stretching between the cell
#                               center and the first land cell on the
#                               bearing indicated by WIND_DIRECTION.
#
#
# <Output_Workspace>            - Designates the output workspace where outputs from the Fetch
#                               Model will be written.
#
#
# <Calculate_Weighted_Fetch>    - Select this option if you wish to calculate a weighted fetch product. 
#                               In order for this to function, there needs to be a weighting designated
#                               in the wind direction weighting percent list for each wind direction specified.
#                               The total weighting percent must equal 100.
#
#
 #
  #
   #
    # -----------------------------------------------------------------------------
    # fetch.py - Calculate the fetch of any point in an open water body
    # Version: 2.0
    # August 26, 2004
    # -----------------------------------------------------------------------------
    # Copyright 2004 David Finlayson, Some Rights Reserved
    #
    # This script is licensed for use under a Creative Commons License.
    # Details about the specific rights granted to you under this license can
    # be found at the following URL:
    #
    # http://creativecommons.org/licenses/by-nc-sa/2.0/
    #
    # -----------------------------------------------------------------------------
    #
    # CONTACT INFO:
    #
    # David Finlayson
    # Marine Geology & Geophysics
    # School of Oceanography
    #
    # NOTES:
    #
    # 1) Fetch lengths are multiples of the input raster cellsize. The
    #    length unit is the same as for cellsize. It is assumed that the
    #    raster is projected and that the cell dimensions are uniform
    #    (data in Geographic coordinate systems will return meaningless
    #    results since latitude and longitude are not uniform length
    #    scales).
    #
    # 2) A negative fetch indicates that the fetch length is unbounded for
    #    this cell. In other words, water extends from the cell center all
    #    the way to the edge of the DEM so it is only possible to
    #    determine a MINIMUM fetch.
    #
    # 3) Due to limitations of the processing algorithm, the accuracy of
    #    the resulting fetch raster is about +/- 6 cell lengths in fetch
    #    length AND +/- 3 cell lengths in coordinate position. The final
    #    step in the processing attempts to fill any resulting gap between
    #    the output raster and the input DEM by using EucAllocation.
    #    Higher resolution land rasters can minimize the problems this
    #    creates at the expense of longer processing times.
    #
    # REFERENCE:
    #
    #   Coastal Engineering Research Center, 1984. Shore Protection
    #   Manual, Vol. I. Vicksburg, Mississippi, Dept. of the Army,
    #   Waterways Experiment Station, Corps of Engineers.
    #
    # -------------------------------------------------------------------------------
   #
  #
 #
#

# Import arcpy module
import arcpy



# Check out any necessary licenses
arcpy.CheckOutExtension("spatial")

import string, os, sys, arcgisscripting, time, array
GP = arcgisscripting.create()
from uwoceanfun import UWGeoprocessor
from math import *
from arcpy import env
from arcpy.sa import *

# Initialize the UWGeoprocessor for some customized GP tools
UW = UWGeoprocessor(GP)

def fetchLength(inFloat, outFloat):
    """ calculates the fetch from top to bottom """
    # Read the header information
    finHeaderName = inFloat[:-3] + "hdr"
    fin = open(finHeaderName, 'r')
    header = fin.readlines()
    fin.close()

    # Store header fields we need for later
    ncols = int(header[0].split()[1])
    nrows = int(header[1].split()[1])
    xllcorner = float(header[2].split()[1])
    yllcorner = float(header[3].split()[1])
    UW.cellsize = float(header[4].split()[1])

    # Apply shift since final output seems to be shifted a bit
    xllcorner = xllcorner
    yllcorner = yllcorner
    header[2] = "xllcorner     %s\n" % str(xllcorner)
    header[3] = "yllcorner     %s\n" % str(yllcorner)

    # Copy header into new float header
    foutHeaderName = outFloat[:-3] + "hdr"
    fout = open(foutHeaderName, 'w')
    fout.writelines(header)
    fout.close()

    # Open data files for processing
    fin = open(inFloat, 'rb')
    fout = open(outFloat, 'wb')

    # The top row of cells in the DEM is read. If these cells are land
    # (elv > 0.0), the fetch distance is set to 0.0, otherwise the
    # cellsize is added to the current fetch distance. Repeat with the
    # next row down the DEM.
    #
    # A negative fetch length value indicates an unbounded fetch. In
    # other words, if water pixels extend all the way to the edge of
    # the DEM, then there is no way to know how large the fetch really
    # is (just a few more meters or across an ocean?). The negative
    # sign makes it easy to identify such cells and treat them
    # differently than bounded fetches.
    
    # Create an array to hold the fetch values of each row. We start
    # with -cellsize assuming that all edge pixels are unbounded until
    # the first land pixel is found.
    fetch = array.array("f", [(-1 * UW.cellsize) for i in range(ncols)])
    
    # Walk down the DEM from the top row
    for row in range(nrows):
        
        # Get the next (first) elevation row from the DEM 
        land = array.array("f")
        land.fromfile(fin, ncols)    ### jjr add

        # Calculate the fetch for this row
        for col in range(ncols):
            if land[col] > 0:
                # Found land, reset fetch length to zero
                fetch[col] = 0
            else:
                # Found water, increment fetch length by cellsize
                if fetch[col] >= 0:
                    # bounded fetch, positive fetch length
                    fetch[col] = fetch[col] + UW.cellsize
                else:
                    # unbounded fetch (so far), negative fetch length
                    fetch[col] = fetch[col] - UW.cellsize

        # Write the fetch for this row out to the new float binary

        fetch.tofile(fout)  ### jjr add

    # Close up the files        
    fin.close()
    fout.close()

def calcSingleSPM(inGrid, outGrid, windDir):
    """ Calculate fetch along one wind direction """

    # Rotate the grid to align with the wind (unless it is already
    # from the North) The grid is rotate Counter Clockwise so that the
    # new grid columns align with the windDir.

    GP.AddMessage("Step 1 of 9: Rotating land raster into alignment with the wind.")
    windDir = (-1 * windDir) % 360.0
    input = inGrid
    output = UW.tempGrid()
    UW.deleteGrid(output)
    pivot = UW.rotateGrid(input, output, windDir)

    # Converting Raster to Float Binary
    GP.AddMessage("Step 2 of 9: Converting raster to floating point binary file.")
    input = output
    output = UW.tempFloat()
    UW.deleteFloat(output)
    UW.rasterToFloat(input, output)
    UW.deleteGrid(input)

    # Calculate the fetch length
    GP.AddMessage("Step 3 of 9: Calculating fetch length.")
    input = output
    output = UW.tempFloat()
    UW.deleteFloat(output)
    fetchLength(input, output)
    UW.deleteFloat(input)

    # Converting Fetch Binary to Raster
    GP.AddMessage("Step 4 of 9: Converting floating point binary fetch to a Grid.")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    UW.floatToRaster(input, output)
    UW.deleteFloat(input)

    # Rotating Raster into North alignment (reverse the rotation
    # performed above)
    GP.AddMessage("Step 5 of 9: Rotating fetch grid back to North alignment.")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    angle = (-1 * windDir) % 360.0
    UW.rotateGrid(input, output, angle, pivot)
    UW.deleteGrid(input)

    # Clipping Final Grid
    GP.AddMessage("Step 6 of 9: Clipping the fetch grid to original extent.")
    input = output
    output = UW.tempGrid()
    clipGrid = inGrid
    UW.deleteGrid(output)
    UW.clipToGrid(input, output, clipGrid)
    UW.deleteGrid(input)

    # Setting fetch length = 0.0 to NODATA
    GP.AddMessage("Step 7 of 9: Setting zero fetches to NODATA")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    GP.SetNull_sa(input, input, output, "Value = 0")
    UW.deleteGrid(input)

    # Convert grid to integer for Allocation process
    GP.AddMessage("Step 8 of 9: Converting fetch grid to integer format")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    GP.Int_sa(input, output)
    UW.deleteGrid(input)

    # Allocate a few cells length buffer around results to make sure we
    # intersect the shoreline
    GP.AddMessage("Step 9 of 9: Expanding results to intersect shoreline")
    input = output
    output = outGrid
    UW.deleteGrid(output)
    distance = str(4.0 * UW.cellsize)
    Output_distance_raster = ""
    Output_direction_raster = ""
    GP.EucAllocation_sa(input, output, distance, "", str(UW.cellsize), "VALUE", Output_distance_raster, Output_direction_raster)
    UW.deleteGrid(input)

def calcSPM(inGrid, windDir, windDirInt, fullpath, windRatFlt, calcWt):
    """ Calculate fetch along one wind direction using 9 radials"""


    startTime = datetime.datetime.now()
    windDirTxt = str(windDir)
    GP.AddMessage("Starting wind direction (" + windDirTxt + ") at: " + str(startTime))
    
    # Calculate the 9 radial directions
    windDirs = []
    for angle in range(-12, 15, 3):
        windDirs.append((windDir + angle) % 360.0)
        
    # Calculate the first fetch length grid
    timeStart = time.time()
    GP.AddMessage("\nTASK 1A of 11: Calculate fetch for radial 1: Bearing %3.0f\n" % windDirs[0])
    input = inGrid

    outFetch = UW.tempGrid()
    UW.deleteGrid(outFetch)
    calcSingleSPM(input, outFetch, windDirs[0])

    # Keep track of which cells have unbounded fetches (negative values)
    GP.AddMessage("\nTASK 1B of 11: Store results for final calculations.")
    GP.AddMessage("Step 1 of 2: Identify unbounded fetches.")
    
    outUnbounded_new = UW.tempGrid()
    UW.deleteGrid(outUnbounded_new)
    inFetch = outFetch
    GP.Con_sa(inFetch, 1, outUnbounded_new, 0, "VALUE < 0")
        
    # Divide the result by 9 (for taking the average of 9 grids)
    GP.AddMessage("Step 2 of 2: Add results to mean fetch total.")
    InExpression = "ABS( %s ) / 9.0" % inFetch
    outAverage_new = UW.tempGrid()
    UW.deleteGrid(outAverage_new)
    GP.SingleOutputMapAlgebra_sa(InExpression, outAverage_new)
    UW.deleteGrid(inFetch)
    timeStop = time.time()

    # Estimate remaining time.
    minutes = ((timeStop - timeStart) / 60.0) * 9
    GP.AddMessage("\nEstimated time remaining: %.0f minutes\n" % minutes) 
    
    # Loop through remaining 8 grids
    for i in range(9)[1:]:
        # Calculate the next fetch length grid
        GP.AddMessage("\nTASK %dA of 11: Calculate fetch for radial %d: Bearing %3.0f\n" % (i+1, i+1, windDirs[i]))
        timeStart = time.time()
        outFetch = UW.tempGrid()
        calcSingleSPM(inGrid, outFetch, windDirs[i])

        # Keep track of which cells have unbounded fetches (negative values)
        # by setting tempNegNames to 1 (true = negative)
        GP.AddMessage("\nTASK %dB of 11: Store results for final calculations.\n" % (i+1))
        GP.AddMessage("Step 1 of 2: Identify unbounded fetches.")
        outUnbounded_old = outUnbounded_new
        outUnbounded_new = UW.tempGrid()
        inFetch = outFetch
        UW.deleteGrid(outUnbounded_new)
        GP.Con_sa(inFetch, 1, outUnbounded_new, outUnbounded_old, "VALUE < 0")
        UW.deleteGrid(outUnbounded_old)
 
        # Divide the abs(result) by 9 and add to earlier results (for taking the average of 9 grids)
        GP.AddMessage("Step 2 of 2: Add results to mean fetch total.")
        outAverage_old = outAverage_new
        outAverage_new = UW.tempGrid()
        InExpression = "%s + ( ABS( %s ) / 9.0 )" % (outAverage_old, inFetch)
        UW.deleteGrid(outAverage_new)
        GP.SingleOutputMapAlgebra_sa(InExpression, outAverage_new)
        UW.deleteGrid(inFetch)
        UW.deleteGrid(outAverage_old)
        timeStop = time.time()

        # Estimate remaining time
        minutes = (timeStop - timeStart) / 60.0 * (9 - i)
        GP.AddMessage("\nEstimated time remaining: %.0f minutes\n" % minutes)
        
    # Reapply the negative sign: step 1 create a "negative" version of
    # results
    GP.AddMessage("\nTASK 10 of 11: Mark unbounded fetches with negative sign.\n")
    InExpression = "-1 * %s" % outAverage_new
    tempNegFile = UW.tempGrid()
    UW.deleteGrid(tempNegFile)
    GP.SingleOutputMapAlgebra_sa(InExpression, tempNegFile)
    
    # Reapply the negative sign: step 2 choose between negative or
    # positive version by looking at the tempNegNames grid
    tempFinal = UW.tempGrid()
    UW.deleteGrid(tempFinal)
    GP.Con_sa(outUnbounded_new, outAverage_new, tempFinal, tempNegFile, "Value = 0")
    UW.deleteGrid(outUnbounded_new)
    UW.deleteGrid(outAverage_new)
    UW.deleteGrid(tempNegFile)

    RasterName = "fet_" + str(windDirInt).zfill(3) + ".tif"
    outRaster = os.path.join(fullpath, RasterName)
    
    # Copy the final temp file into the finished directory and rename
    GP.AddMessage("\nTASK 11 of 11: Copying final fetch raster to %s.\n" % outRaster)


    tempMask = UW.tempGrid()
    UW.deleteGrid(tempMask)
    GP.Con_sa(inGrid, "1", tempMask, "", "Value = 0")
    
    tempClip = UW.tempGrid()
    UW.deleteGrid(tempClip)
    GP.ExtractByMask_sa(tempFinal, tempMask, tempClip)
    
    GP.SingleOutputMapAlgebra_sa("INT ( " + tempClip + " + 0.5)", outRaster, "")

    ###
    if calcWt == "true":
        tempfetf = UW.tempGrid()
        UW.deleteGrid(tempfetf)
        
        GP.AddMessage("Calculating weighted wind fetch component for direction: " + str(windDirInt))
        GP.SingleOutputMapAlgebra_sa(outRaster + " * " + str(windRatFlt),tempfetf, "")
        
        RasterNameWt = "fetw_" + str(windDirInt).zfill(3)
        outRasterWt = os.path.join(fullpath, RasterNameWt)

        GP.Int_sa(tempfetf, outRasterWt)
        UW.deleteGrid(tempfetf)
    
        
    GP.AddMessage("Clean up the remaining temp files")
    # Clean up the remaining temp files
    UW.deleteGrid(tempFinal)
    UW.deleteGrid(tempMask)
    UW.deleteGrid(tempClip)

    arcpy.env.workspace = fullpath

    # Get and print a list of GRIDs from the workspace
    gg_rasters = arcpy.ListRasters("g_g*", "GRID")
    for gg_raster in gg_rasters:
        if arcpy.Exists(fullpath + "\\" + str(gg_raster)): 
            arcpy.Delete_management(fullpath + "\\" + str(gg_raster))


    endTime = datetime.datetime.now()
    GP.AddMessage("Totally completed wind direction (" + windDirTxt + ") successfully in: " + str(endTime - startTime))
        
    
    GP.AddMessage("____________________________________________________")

def calcSPM_Res(inGrid, windDir, windDirInt, fullpath, windRatFlt, calcWt):
    """ Calculate fetch along one wind direction using 5 radials"""
    
    # Calculate the 5 radial directions
    windDirs = []
    for angle in range(-6, 9, 3):
        windDirs.append((windDir + angle) % 360.0)
        
    # Calculate the first fetch length grid
    timeStart = time.time()
    GP.AddMessage("\nTASK 1A of 7: Calculate fetch for radial 1: Bearing %3.0f\n" % windDirs[0])
    input = inGrid

    outFetch = UW.tempGrid()
    UW.deleteGrid(outFetch)
    calcSingleSPM(input, outFetch, windDirs[0])

    # Keep track of which cells have unbounded fetches (negative values)
    GP.AddMessage("\nTASK 1B of 7: Store results for final calculations.")
    GP.AddMessage("Step 1 of 2: Identify unbounded fetches.")
    
    outUnbounded_new = UW.tempGrid()
    UW.deleteGrid(outUnbounded_new)
    inFetch = outFetch
    GP.Con_sa(inFetch, 1, outUnbounded_new, 0, "VALUE < 0")
        
    # Divide the result by 5 (for taking the average of 5 grids)
    GP.AddMessage("Step 2 of 2: Add results to mean fetch total.")
    InExpression = "ABS( %s ) / 5.0" % inFetch
    outAverage_new = UW.tempGrid()
    UW.deleteGrid(outAverage_new)
    GP.SingleOutputMapAlgebra_sa(InExpression, outAverage_new)
    UW.deleteGrid(inFetch)
    timeStop = time.time()

    # Estimate remaining time.
    minutes = ((timeStop - timeStart) / 60.0) * 5
    GP.AddMessage("\nEstimated time remaining: %.0f minutes\n" % minutes) 
    
    # Loop through remaining 4 grids
    for i in range(5)[1:]:
        # Calculate the next fetch length grid
        GP.AddMessage("\nTASK %dA of 7: Calculate fetch for radial %d: Bearing %3.0f\n" % (i+1, i+1, windDirs[i]))
        timeStart = time.time()
        outFetch = UW.tempGrid()
        calcSingleSPM(inGrid, outFetch, windDirs[i])

        # Keep track of which cells have unbounded fetches (negative values)
        # by setting tempNegNames to 1 (true = negative)
        GP.AddMessage("\nTASK %dB of 7: Store results for final calculations.\n" % (i+1))
        GP.AddMessage("Step 1 of 2: Identify unbounded fetches.")
        outUnbounded_old = outUnbounded_new
        outUnbounded_new = UW.tempGrid()
        inFetch = outFetch
        UW.deleteGrid(outUnbounded_new)
        GP.Con_sa(inFetch, 1, outUnbounded_new, outUnbounded_old, "VALUE < 0")
        UW.deleteGrid(outUnbounded_old)
 
        # Divide the abs(result) by 5 and add to earlier results (for taking the average of 5 grids)
        GP.AddMessage("Step 2 of 2: Add results to mean fetch total.")
        outAverage_old = outAverage_new
        outAverage_new = UW.tempGrid()
        InExpression = "%s + ( ABS( %s ) / 5.0 )" % (outAverage_old, inFetch)
        UW.deleteGrid(outAverage_new)
        GP.SingleOutputMapAlgebra_sa(InExpression, outAverage_new)
        UW.deleteGrid(inFetch)
        UW.deleteGrid(outAverage_old)
        timeStop = time.time()

        # Estimate remaining time
        minutes = (timeStop - timeStart) / 60.0 * (5 - i)
        GP.AddMessage("\nEstimated time remaining: %.0f minutes\n" % minutes)
        
    # Reapply the negative sign: step 1 create a "negative" version of
    # results
    GP.AddMessage("\nTASK 6 of 7: Mark unbounded fetches with negative sign.\n")
    InExpression = "-1 * %s" % outAverage_new
    tempNegFile = UW.tempGrid()
    UW.deleteGrid(tempNegFile)
    GP.SingleOutputMapAlgebra_sa(InExpression, tempNegFile)
    
    # Reapply the negative sign: step 2 choose between negative or
    # positive version by looking at the tempNegNames grid
    tempFinal = UW.tempGrid()
    UW.deleteGrid(tempFinal)
    GP.Con_sa(outUnbounded_new, outAverage_new, tempFinal, tempNegFile, "Value = 0")
    UW.deleteGrid(outUnbounded_new)
    UW.deleteGrid(outAverage_new)
    UW.deleteGrid(tempNegFile)

    RasterName = "fet_" + str(windDirInt).zfill(3) + ".tif"
    outRaster = os.path.join(fullpath, RasterName)
    
    # Copy the final temp file into the finished directory and rename
    GP.AddMessage("\nTASK 7 of 7: Copying final fetch raster to %s.\n" % outRaster)

    tempMask = UW.tempGrid()
    UW.deleteGrid(tempMask)
    GP.Con_sa(inGrid, "1", tempMask, "", "Value = 0")
    
    tempClip = UW.tempGrid()
    UW.deleteGrid(tempClip)
    GP.ExtractByMask_sa(tempFinal, tempMask, tempClip)
    
    GP.SingleOutputMapAlgebra_sa("INT ( " + tempClip + " + 0.5)", outRaster, "")

    ###
    if calcWt == "true":
        tempfetf = UW.tempGrid()
        UW.deleteGrid(tempfetf)
        
        GP.AddMessage("Calculating weighted wind fetch component for direction: " + str(windDirInt))
        GP.SingleOutputMapAlgebra_sa(outRaster + " * " + str(windRatFlt),tempfetf, "")
        
        RasterNameWt = "fetw_" + str(windDirInt).zfill(3)
        outRasterWt = os.path.join(fullpath, RasterNameWt)

        GP.Int_sa(tempfetf, outRasterWt)
        UW.deleteGrid(tempfetf)
        
    GP.AddMessage("Clean up the remaining temp files")
    # Clean up the remaining temp files
    UW.deleteGrid(tempFinal)
    UW.deleteGrid(tempMask)
    UW.deleteGrid(tempClip)

    arcpy.env.workspace = fullpath

    # Get and print a list of GRIDs from the workspace
    gg_rasters = arcpy.ListRasters("g_g*", "GRID")
    for gg_raster in gg_rasters:
        if arcpy.Exists(fullpath + "\\" + str(gg_raster)): 
            arcpy.Delete_management(fullpath + "\\" + str(gg_raster))
    
    GP.AddMessage("____________________________________________________")
    
def calcSingle(inGrid, windDir, windDirInt, fullpath, windRatFlt, calcWt):
    """ Calculate fetch along one wind direction """

    # Rotate the grid to align with the wind (unless it is already
    # from the North) The grid is rotate Counter Clockwise so that the
    # new grid columns align with the windDir.

    startTime = datetime.datetime.now()
    windDirTxt = str(windDir)
    GP.AddMessage("Starting wind direction (" + windDirTxt + ") at: " + str(startTime))

    GP.AddMessage("Step 1 of 9: Rotating land raster into alignment with the wind.")
    windDir = (-1 * windDir) % 360.0
    input = inGrid
    output = UW.tempGrid()
    UW.deleteGrid(output)
    pivot = UW.rotateGrid(input, output, windDir)

    # Converting Raster to Float Binary
    GP.AddMessage("Step 2 of 9: Converting raster to floating point binary file.")
    input = output
    output = UW.tempFloat()
    UW.deleteFloat(output)
    UW.rasterToFloat(input, output)
    UW.deleteGrid(input)

    # Calculate the fetch length
    GP.AddMessage("Step 3 of 9: Calculating fetch length.")
    input = output
    output = UW.tempFloat()
    UW.deleteFloat(output)
    fetchLength(input, output)
    UW.deleteFloat(input)

    # Converting Fetch Binary to Raster
    GP.AddMessage("Step 4 of 9: Converting floating point binary fetch to a Grid.")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    UW.floatToRaster(input, output)
    UW.deleteFloat(input)

    # Rotating Raster into North alignment (reverse the rotation
    # performed above)
    GP.AddMessage("Step 5 of 9: Rotating fetch grid back to North alignment.")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    angle = (-1 * windDir) % 360.0
    UW.rotateGrid(input, output, angle, pivot)
    UW.deleteGrid(input)

    # Clipping Final Grid
    GP.AddMessage("Step 6 of 9: Clipping the fetch grid to original extent.")
    input = output
    output = UW.tempGrid()
    clipGrid = inGrid
    UW.deleteGrid(output)
    UW.clipToGrid(input, output, clipGrid)
    UW.deleteGrid(input)

    # Setting fetch length = 0.0 to NODATA
    GP.AddMessage("Step 7 of 9: Setting zero fetches to NODATA")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    GP.SetNull_sa(input, input, output, "Value = 0")
    UW.deleteGrid(input)

    # Convert grid to integer for Allocation process
    GP.AddMessage("Step 8 of 9: Converting fetch grid to integer format")
    input = output
    output = UW.tempGrid()
    UW.deleteGrid(output)
    GP.Int_sa(input, output)
    UW.deleteGrid(input)

    # Allocate a few cells length buffer around results to make sure we
    # intersect the shoreline
    GP.AddMessage("Step 9 of 9: Expanding results to intersect shoreline")
    input = output

    RasterName = "fet_" + str(windDirInt).zfill(3) + ".tif"
    outRaster = os.path.join(fullpath, RasterName)
    
                               
    tempFinal = UW.tempGrid()
    UW.deleteGrid(tempFinal)
    distance = str(4.0 * UW.cellsize)
    Output_distance_raster = ""
    Output_direction_raster = ""
    GP.EucAllocation_sa(input, tempFinal, distance, "", str(UW.cellsize), "VALUE", Output_distance_raster, Output_direction_raster)
    UW.deleteGrid(input)
    
    # Copy the final temp file into the finished directory and rename
    GP.AddMessage("\nCopying final fetch raster to %s.\n" % outRaster)
    

    tempMask = UW.tempGrid()
    UW.deleteGrid(tempMask)
    GP.Con_sa(inGrid, "1", tempMask, "", "Value = 0")
    
    tempClip = UW.tempGrid()
    UW.deleteGrid(tempClip)
    GP.ExtractByMask_sa(tempFinal, tempMask, tempClip)

    GP.SingleOutputMapAlgebra_sa("INT ( " + tempClip + " + 0.5)", outRaster, "")


    if calcWt == "true":
        tempfetf = UW.tempGrid()
        
        GP.AddMessage("Calculating weighted wind fetch component for direction: " + str(windDirInt))
        GP.SingleOutputMapAlgebra_sa(outRaster + " * " + str(windRatFlt),tempfetf, "")

        
        RasterNameWt = "fetw_" + str(windDirInt).zfill(3)
        outRasterWt = os.path.join(fullpath, RasterNameWt)

        GP.Int_sa(tempfetf, outRasterWt)
        UW.deleteGrid(tempfetf)
        
    GP.AddMessage("Clean up the remaining temp files")
    
    # Clean up the remaining temp files
    
    UW.deleteGrid(tempFinal)
    UW.deleteGrid(tempMask)
    UW.deleteGrid(tempClip)


    arcpy.env.workspace = fullpath

    # Get and print a list of GRIDs from the workspace
    gg_rasters = arcpy.ListRasters("g_g*", "GRID")
    for gg_raster in gg_rasters:
        if arcpy.Exists(fullpath + "\\" + str(gg_raster)): 
            arcpy.Delete_management(fullpath + "\\" + str(gg_raster))

    
    endTime = datetime.datetime.now()
    GP.AddMessage("Totally completed wind direction (" + windDirTxt + ") successfully in: " + str(endTime - startTime))
    
    
    GP.AddMessage("____________________________________________________")
    
def expandExtent(inGrid):
    
    ### expand extent start ###
    # This snippet of code was added to insure that when rotating the grids to calculate fetch
    # the extent used was a perfect square, thus making certain that for instance, a grid with
    # an extent that was significantly longer North to South than it is East to West, when you
    # calculate fetch at 90 degrees the extent of the resulting raster is wide enough from
    # east to west to approximate the rotation of the grid.
    
    dsc = GP.Describe(inGrid)
    
    cellSize1 = float(dsc.meancellheight)

    extentList = dsc.Extent.split(" ")
    XMin1 = float(extentList[0])
    YMin1 = float(extentList[1])
    XMax1 = float(extentList[2])
    YMax1 = float(extentList[3])
   
    XDiff = XMax1 - XMin1
    YDiff = YMax1 - YMin1
    
 
    gridCenterX = (XMin1 + (XDiff / 2))
    gridCenterY = (YMin1 + (YDiff / 2))


    hypDiff = pow(((XDiff * XDiff) + (YDiff * YDiff)), 0.5)
    hypDiff = (hypDiff / 2)

    hypDiff2 = int((hypDiff / cellSize1) + 1)
    hypDiff3 = hypDiff2 * cellSize1
    
    XMin2 = gridCenterX - hypDiff3
    YMin2 = gridCenterY - hypDiff3
    XMax2 = gridCenterX + hypDiff3
    YMax2 = gridCenterY + hypDiff3
        
    GP.AddMessage("Adjusted analysis extent to accommodate rotation: XMin = " + str(XMin2) + " YMin = " + str(YMin2) + " XMax = " + str(XMax2) + " YMax = " + str(YMax2))
    GP.AddMessage("____________________________________________________")
    
    extentString = str(XMin2) + " " + str(YMin2) + " " + str(XMax2) + " " + str(YMax2)
    GP.Extent = extentString
    GP.SnapRaster = inGrid
    
def main():
    
    # Get the script parameters    
    if len(sys.argv) < 5:
        GP.AddError("Fetch.py did not receive all five arguments")
        sys.exit(1)
    else:
        inGrid = sys.argv[1]
        inputFile = sys.argv[2]
        calcMethod = sys.argv[3]
        outWS = sys.argv[4]
        calcWt = sys.argv[5]

        
    tempDir = 0
    
    while True:
        tempDir = tempDir + 1
        dirName = "fetch" + str(tempDir).zfill(3)
        fullpath = os.path.join(outWS, dirName)

        if not GP.Exists(fullpath):
            break
        
    GP.ScratchWorkspace = fullpath  
    GP.CreateFolder_management(outWS, dirName)
    GP.AddMessage("Creating directory " + dirName + " at " + outWS)

    expandExtent(inGrid)
       
    # Process fetch length
    
    if calcWt == "true":
        rasterCalcWt = ""

        inpCheck = open(inputFile,"r")
        for inputLineCheck in inpCheck:
            windRatioCheck = inputLineCheck.split(",")[1] # retrieve wind direction ratio from textfile
            windRatioCheck = windRatioCheck.strip()
            
            windRatCheckFlt = float(windRatioCheck)


            if rasterCalcWt == "":
                rasterCalcWt = windRatCheckFlt
            else:
                rasterCalcWt = rasterCalcWt + windRatCheckFlt
            
        inpCheck.close()
       
        if rasterCalcWt < 99.6 or rasterCalcWt > 100.4: # accounts for rounding errors/data precision
            GP.AddError("Total weighted wind percentage equals: [" + str(rasterCalcWt) + "] Update wind direction list so sum is [100]")
            sys.exit(0)


    rasterCalcWt2 = ""
    
    inp = open(inputFile,"r")
    for inputLine in inp:
        
        windRatFlt = 0
        
        windDir = inputLine.split(",")[0] # retrieve wind direction from textfile
        windDir = windDir.strip()
        windDir = float(windDir) % 360.0
        windDirInt = int(windDir)
        
        GP.AddMessage("Input wind direction (degrees): " + str(windDir))
        
        if calcWt == "true":
            windRatio = inputLine.split(",")[1] # retrieve wind direction ratio from textfile
            windRatio = windRatio.strip()

            windRatFlt = float(windRatio)
            
            rasterNameWt = "fetw_" + str(windDirInt).zfill(3)
            outRasterWt = os.path.join(fullpath, rasterNameWt)
                
            if rasterCalcWt2 == "":
                rasterCalcWt2 = outRasterWt
            else:
                rasterCalcWt2 = rasterCalcWt2 + " + " + outRasterWt
            
            GP.AddMessage("Input wind allocation (percent): " + str(windRatFlt))
        
        if calcMethod == "SINGLE":
            try:
                calcSingle(inGrid, windDir, windDirInt, fullpath, windRatFlt, calcWt)
            except:
                GP.AddError(GP.GetMessages(2))
                raise
        elif calcMethod == "SPM":
            try:
                calcSPM(inGrid, windDir, windDirInt, fullpath, windRatFlt, calcWt)
            except:
                GP.AddError(GP.GetMessages(2))
                raise
        elif calcMethod == "SPM-Restricted":
            try:
                calcSPM_Res(inGrid, windDir, windDirInt, fullpath, windRatFlt, calcWt)
            except:
                GP.AddError(GP.GetMessages(2))
                raise
        else:
            GP.AddError("Invalid calculation method: %s, must be either 'SINGLE' or 'SPM'." % calcMethod)
            sys.exit(0)
        
    inp.close()
            
    outRasterWtFin = os.path.join(fullpath, "fetw_sum.tif")
    

    if calcWt == "true":
        GP.AddMessage("Calculating weighted wind fetch final output [" + outRasterWtFin + "]")
        GP.SingleOutputMapAlgebra_sa("(" + rasterCalcWt2 + ") / 100",outRasterWtFin, "")

    inpDel = open(inputFile,"r")
    for inputDelLine in inpDel:
        
        windDir = inputDelLine.split(",")[0] # retrieve wind direction from textfile
        windDir = windDir.strip()
        windDir = float(windDir) % 360.0
        windDirInt = int(windDir)
        rasterNameWt = "fetw_" + str(windDirInt).zfill(3)
        
        rasterNameWtFull = os.path.join(fullpath, rasterNameWt)
        
        UW.deleteGrid(rasterNameWtFull)
        
    inpDel.close()

    for temp_file in arcpy.ListFiles("temp*"):

        arcpy.Delete_management(fullpath + "\\" + str(temp_file))

    
    
if __name__ == "__main__":
    main()
        
