# REQUIREMENTS:
#
#  - ESRI ArcGIS 10.0/10.1
#  - Spatial Analyst Extension
#  - uwoceanfun.py (a library of helper functions developed by David Finlayson, USGS-
#    Pacific Science Center)
#
#   This model calculates several wave characteristic outputs including; wave height, wave length, 
#   wave period, shear stress, and maximum orbital wave velocity.
#
# USAGE:
#
# <Input_Wind_Fetch_Workspace>
#
#           Full path to a workspace that contains pre-constructed wind fetch raster datasets to
#           be used in wave models. Fetch should be calculated in Meters.
#
#
# <Wind_Direction_and_Speed_List__Dir__Spd__Date_>
#
#           Specific wind parameter used was the maximum 2-minute average wind speed and direction
#           (in miles per hour and degrees, respectively). The text file of collected wind data is
#           contained as comma-delimited numeric values consisting of the wind direction, followed
#           by the wind speed, and finally the date of data collection expressed as a two-digit year,
#           followed by a two-digit month, and finally the two-digit day (e.g. 020421 = April 21, 2002). 
#
#           It's important the date values be organized like this for the model to work correctly. 
#
#           The assembled wind speed data is adjusted to approximate a one hour wind duration, a
#           ten meter anemometer height above the ground surface, and also adjusted for coefficient
#           of drag. These adjustments directly affect the input parameters of wave height, period,
#           and length.
#
#
# <Wind_Measurement_Height_Above_Ground__Meters_>
#
#           Adjustments to the wind speed data may need to be made to better approximate
#           real-world conditions above water. Wind data for previous examples was
#           collected from the National Oceanic and Atmospheric Administration, National
#           Climatic Data Center (http://www7.ncdc.noaa.gov/IPS/lcd/lcd.html).
#           Specific wind parameter used was the maximum 2-minute average wind speed and direction
#           (in miles per hour and degrees, respectively). The wind speed data collected are adjusted
#           to approximate a ten meter anemometer height above the ground surface using the input within
#           the model dialog entitled "Wind Measurement Height Above Ground (Meters)". The 10 meter
#           elevation measurement guideline is established within the Automated Surface Observing
#           System (ASOS) specifications. If however, the data collected was from an anemometer at an
#           elevation other than 10 meters the following algorithm would have been applied:
#
#
# <Density_of_Water__kg_m3_>
#
#           Default is 998.2 kg/m3. User can also enter path to a raster denoting surface of varying water density.
#
#
# <Bathymetry_Raster__Meters_>
#
#           Bathymetric data used to calculate wave models. Depth values in ESRI raster
#           should be in meters.
#
#
# <Output_Workspace>
#
#           Workspace where outputs will be stored. A new directory will be created entitled
#           "Wave---" where "---" is an integer incremented from previous model run.
#
#
# <Calculate Percentage of Days Sediment is Suspended>
#
#           Check this box if you wish to calculate the percentage of days that sediment   
#           is predicted to be suspended according to the MOWV threshold selected.
#           This is done by overlaying and adding up the values of the sediment suspension
#           probability rasters and dividing by the total number of days analyzed.
#
#
#   Be sure to set your Scratchworkspace in your processing
#   Environment before running the script!  Do not use a geodatabase for this.
#
# REFERENCE:
#
#   Coastal Engineering Research Center, 1984. Shore Protection
#   Manual, Vol. I. Vicksburg, Mississippi, Dept. of the Army,
#   Waterways Experiment Station, Corps of Engineers.
#
#   U.S. Army Corps of Engineers, 2002, Coastal Engineering Manual, Engineer 
#   Manual 1110-2-1100, U.S. Army Corps of Engineers, Washington, D.C.
#   (in 6 volumes).
#
# -------------------------------------------------------------------------------

# Import arcpy module
import arcpy

# Check out any necessary licenses
arcpy.CheckOutExtension("spatial")

import string, os, sys, arcgisscripting, time, array
GP = arcgisscripting.create()
from uwoceanfun import UWGeoprocessor
from math import *

# Initialize the UWGeoprocessor for some customized GP tools
UW = UWGeoprocessor(GP)

def calcWave(inGrid, windDir, windDirInt, windSpd, fullpath, bathGrid, WatDensity, IndWindDate):
    
    NDFetchRasterName = "ndf_" + IndWindDate
    outRasterNdf = os.path.join(fullpath, NDFetchRasterName)
    NDHeightRasterName = "ndh_" + IndWindDate
    outRasterNdh = os.path.join(fullpath, NDHeightRasterName)    
    HgtRasterName = "hgt_" + IndWindDate
    outRasterHgt = os.path.join(fullpath, HgtRasterName)
    PerRasterName = "per_" + IndWindDate
    outRasterPer = os.path.join(fullpath, PerRasterName)
    LenRasterName = "len_" + IndWindDate
    outRasterLen = os.path.join(fullpath, LenRasterName)
    VelRasterName = "vel_" + IndWindDate
    outRasterVel = os.path.join(fullpath, VelRasterName)
    SsRasterName = "str_" + IndWindDate
    outRasterSs = os.path.join(fullpath, SsRasterName)
    VelRasterName2 = "vec_" + IndWindDate
    outRasterVel2 = os.path.join(fullpath, VelRasterName2)

    NdpRasterName = "np1_" + IndWindDate
    outRasterNdp = os.path.join(fullpath, NdpRasterName)
    
    ### ------------------------------------
    ### Starting wave height calculation
    GP.AddMessage("\nCreating wave height raster: %s\n" % outRasterHgt)
    
    CoefDrag = 0.001 * (1.1 + (0.035 * windSpd))
    FrictionVel = sqrt(CoefDrag) * windSpd
    GP.SingleOutputMapAlgebra_sa("(9.82 * " + inGrid + ") / POW(" + str(FrictionVel) + ", 2)", outRasterNdf, "") ### Wave Height
    GP.SingleOutputMapAlgebra_sa("0.0413 * SQRT( " + outRasterNdf + ")", outRasterNdh, "")
    GP.SingleOutputMapAlgebra_sa("(" + outRasterNdh + " * (POW(" + str(FrictionVel) + ", 2) / 9.82))", outRasterHgt, "")
    ### Finished Wave Height Calculation
        
    ### ------------------------------------
    ### Starting wave period calculation
    GP.AddMessage("\nCreating wave period raster: %s\n" % outRasterPer)

    GP.SingleOutputMapAlgebra_sa("0.751 * (POW(" + outRasterNdf + ", 0.33333333333))", outRasterNdp, "")### Non-dimensional Wave Period1
    GP.SingleOutputMapAlgebra_sa("(" + outRasterNdp + " * " + str(FrictionVel) + ") / 9.82", outRasterPer, "")### Non-dimensional Wave Period2
    
    ### Finished Wave period Calculation
    
    ### ------------------------------------
    ### Starting wave length calculation
    GP.AddMessage("\nCreating wave length raster: %s\n" % outRasterLen)

    GP.SingleOutputMapAlgebra_sa("9.82 * (POW(" + outRasterPer + ", 2)) / (2 * 3.14159265)", outRasterLen, "")### Wave Length
    ### Finished Wave length Calculation
    
    ### ------------------------------------
    ### Starting max. orbital wave velocity calculation
    GP.AddMessage("\nCreating orbital wave velocity raster: %s\n" % outRasterVel)

    GP.SingleOutputMapAlgebra_sa("3.14159265 * " + outRasterHgt + " / (" + outRasterPer + " * SINH(2 * 3.14159265 * " + bathGrid + " / " + outRasterLen + "))", outRasterVel, "")### Orbital Wave Velocity

    if sys.argv[13] == "true":
        GP.AddMessage("\nCreating sediment suspension probability raster: %s\n" % outRasterVel2)
        GP.Con_sa(outRasterVel, 1, outRasterVel2, 0, "Value >= " + sys.argv[14])### sediment suspension probability
        
    ### Finished max. orbital wave velocity Calculation
    
    ### ------------------------------------
    ### Starting shear stress calculation
    if sys.argv[12] == "true":
        GP.AddMessage("\nCreating shear stress raster: %s\n" % outRasterSs)
        GP.AddMessage("\nCreating shear stress raster: " + str(WatDensity) + " * 0.032 * SQR(" + str(outRasterVel) + ") / 2")
        GP.SingleOutputMapAlgebra_sa(WatDensity + " * 0.032 * SQR(" + outRasterVel + ") / 2", outRasterSs, "")### Shear Stress
        
    ### Finished Shear stress Calculation
    
    ### delete grids for space saving - JJR ############################################

    if sys.argv[8] == "false":
        UW.deleteGrid(outRasterHgt)
    else:
        GP.AddMessage("Wave height raster saved permanently to disk")

    if sys.argv[9] == "false":
        UW.deleteGrid(outRasterPer)
    else:
        GP.AddMessage("Wave period raster saved permanently to disk")

    if sys.argv[10] == "false":
        UW.deleteGrid(outRasterLen)
    else:
        GP.AddMessage("Wave length raster saved permanently to disk")
        
    if sys.argv[11] == "false":
        UW.deleteGrid(outRasterVel)
    else:
        GP.AddMessage("Maximum orbital wave velocity raster saved permanently to disk")

    if sys.argv[12] == "true":
        GP.AddMessage("Shear stress raster saved permanently to disk")

    if sys.argv[13] == "true":
        GP.AddMessage("Sediment suspension probability raster saved permanently to disk")

    UW.deleteGrid(outRasterNdf)
    UW.deleteGrid(outRasterNdh)
    UW.deleteGrid(outRasterNdp)
    

def main():

    # Get the script parameters    
    if len(sys.argv) < 8:
        GP.AddError("wave_jjr.py did not receive all six arguments")
        sys.exit(1)
    else:
        inFetWS = sys.argv[1]
        inputFile = sys.argv[2]
        windHeight = sys.argv[3]
        overLand = sys.argv[4]
        WatDensity = sys.argv[5]
        bathGrid = sys.argv[6]
        outWS = sys.argv[7]
        calcWt = sys.argv[15]

    # ScratchWorkspace
    if GP.ScratchWorkspace == "":
        GP.AddError("A scratch workspace must be defined in your Environmental settings for this script to function properly.")
        sys.exit(0)

    if calcWt == "true" and sys.argv[13] == "false":
        GP.AddError("Cannot calculate percentage of days sediment is suspended output raster if individual sediment suspension probability outputs were not created!")
        sys.exit(0)
        
    tempDir = 0
    
    while True:
        tempDir = tempDir + 1
        dirName = "wave" + str(tempDir).zfill(3)
        fullpath = os.path.join(outWS, dirName)

        if not GP.Exists(fullpath):
            break

    GP.CreateFolder_management(outWS, dirName)
    GP.AddMessage("Creating directory " + dirName + " at " + outWS)
    
    rasterCalcWt2 = ""
    rasterCnt = 0
        
    ### derive wind data parameters from textfile
    inp = file(inputFile,"r")
    for inputLine in inp:

        IndWindDir = inputLine.split(",")[0] # retrieve wind direction from textfile
        IndWindDir = IndWindDir.strip()
        IndWindSpd = inputLine.split(",")[1] # retrieve wind speed from textfile
        IndWindSpd = IndWindSpd.strip()
        IndWindDate = inputLine.split(",")[2] # retrieve date from textfile
        IndWindDate = IndWindDate.strip()
        
        windDir = float(IndWindDir) % 360.0
        windSpd = float(IndWindSpd)
        
        GP.AddMessage("____________________________________________________")
        GP.AddMessage("Input wind direction (degrees): " + str(windDir))
        GP.AddMessage("Unadjusted wind speed (mph): " + str(windSpd))
        windSpd = windSpd * 0.44704 ### conversion to meters per second
        GP.AddMessage("Unadjusted wind speed (meters per second): " + str(windSpd))
        # Starting adjusting for 10 meter elevation (z must be less than 20 meters)
        windHeight = float(windHeight)
        heightAdj = 10 / windHeight

        windHeightAdj = windSpd * pow(heightAdj, 0.142857142)
        GP.AddMessage("Wind speed adjusted for Elevation (meters per second): " + str(windHeightAdj))
        
        windSpd = windHeightAdj
        # Finished adjusting for 10 meter elevation

        
        # Start adjusting fastest mile wind speed to 1 hour average windspeed
        oneMileSec = 3600 / windSpd
        
        windConversion = 1.277 + 0.296 * tanh(0.9 * log10(45 / oneMileSec))
        
        newWindSpd = windSpd / windConversion
        GP.AddMessage("Wind speed adjusted for 1-hour Wind Duration (meters per second): " + str(newWindSpd))

        windSpd = newWindSpd
        # Finished adjusting fastest mile wind speed to 1 hour average windspeed


        # Starting adjusting for overland wind speed measurement
        if overLand == "true":
            windSpd = windSpd * 1.2
            GP.AddMessage("Wind speed adjusted for overland wind speed measurement (meters per second): " + str(windSpd))
        # Finished adjusting for overland wind speed measurement
                
        windDirInt = int(windDir)

        RasterName = "fet_" + str(windDirInt).zfill(3)
        inGrid = os.path.join(inFetWS, RasterName)
        
        SSPRasterNameWt = "vec_" + IndWindDate
        outRasterSSP = os.path.join(fullpath, SSPRasterNameWt)
        
        rasterCnt = rasterCnt + 1
                
        if rasterCalcWt2 == "":
            rasterCalcWt2 = outRasterSSP
        else:
            rasterCalcWt2 = rasterCalcWt2 + " + " + outRasterSSP
                 
        if GP.Exists(inGrid):
            try:
                calcWave(inGrid, windDir, windDirInt, windSpd, fullpath, bathGrid, WatDensity, IndWindDate)
            except:
                GP.AddError(GP.GetMessages(2))
                raise
        else:
            GP.AddError("Invalid Fetch Grid: %s.  Fetch grid does not exist." % inGrid)
            sys.exit(0)
    inp.close()

    outRasterWtFin = os.path.join(fullpath, "susp_prob")
    
    if calcWt == "true" and sys.argv[13] == "true":
        GP.AddMessage("Calculating percentage of days sediment is suspended output [" + outRasterWtFin + "]")
        GP.SingleOutputMapAlgebra_sa("((float(" + rasterCalcWt2 + ") / " + str(rasterCnt) + ") * 100)",outRasterWtFin, "")

        
if __name__ == "__main__":
    main()
    
   
    
    

                  
    



    
    
        
    
            


        
