# REQUIREMENTS:
#
#  - ESRI ArcGIS 10.0/10.1
#  - Spatial Analyst Extension
#  - uwoceanfun.py (a library of helper functions developed by David Finlayson, USGS-
#    Pacific Science Center)
#
#   This tool allows the user to identify the areas where the difference between the values 
#   from two raster datasets are above a certain threshold (magnitude). For example, if the
#   user wants to identify areas where there is at least a 10 percent greater chance that sediment
#   will be suspended when comparing alternative management scenarios it will generate a shapefile 
#   identifying those areas.
#
# USAGE:
#
# <Input_Raster>
#
#           Full path to an ESRI raster data set
#
#
# <Comparison_Raster>
#
#           Full path to an ESRI raster data set
#
#
# <Magnitude_of_Difference__areas_larger_than_this_difference_will_be_identified_>
#
#           A number identifying a desired magnitude of difference from subtracting the values from
#           two raster datasets. For example, if the user wants to identify areas where there is
#           at least a 10 percent greater chance that sediment will be suspended when comparing
#           alternative management scenarios or entering a wind fetch length and identifying where
#           there are areas that exceed this difference threshold when comparing alternative scenarios.
#           
#
# <Output_Shapefile>
#
#           Full path to an ESRI shapefile data set depicting the polygonal areas that meet the defined
#           magnitude of difference threshold.  Use the field GRIDCODE to identify areas meeting the 
#           specified magnitude of difference.  Polygons with a GRIDCODE of "1" have a greater magnitude 
#           of difference than that specified and those polygons with a GRIDCODE of "0" have a smaller or
#           equal magnitude of difference than that specified.
#
#
# <Output_Raster (Optional)>
#
#           Full path to an output ESRI raster data set depicting the difference values between the two
#           input raster data sets.  This is an optional output.
#
#
#   Be sure to set your Scratchworkspace in your processing
#   Environment before running the script!  Do not use a geodatabase for this.
#
#
# -------------------------------------------------------------------------------
# Import arcpy module
import arcpy

# Check out any necessary licenses
arcpy.CheckOutExtension("spatial")

import string, os, sys, arcgisscripting, time, array
GP = arcgisscripting.create()
from uwoceanfun import UWGeoprocessor
from math import *

# Initialize the UWGeoprocessor for some customized GP tools
UW = UWGeoprocessor(GP)

def calcAPE(inRast1, inRast2, apeValue, outShape, outRast):
    
    outputMinus = UW.tempGrid()

    #subtract one grid from the other
    arcpy.gp.Minus_sa(inRast1, inRast2, outputMinus)
    
    if outRast == "#":
        outputInt = UW.tempGrid()
    else:
        outputInt = outRast
        
    #convert this output to an integer grid
    arcpy.gp.Int_sa(outputMinus, outputInt)

    #find the maximum difference between both grids
    
    arcpy.CalculateStatistics_management(outputInt)
    GP.AddMessage("calc stats for " + outputInt)
    
    absMaxRes = arcpy.GetRasterProperties_management(outputInt, "MAXIMUM")#.GetOutput(0)
    absMinRes = arcpy.GetRasterProperties_management(outputInt, "MINIMUM")
       
    absMaxStr = str(absMaxRes)
    absMinStr = str(absMinRes)
    
    absMaxFlt = float(absMaxStr)
    absMinFlt = float(absMinStr)

    abs2MaxFlt = abs(absMaxFlt)
    abs2MinFlt = abs(absMinFlt)

    if abs2MaxFlt > abs2MinFlt:
        absMaxAll = abs2MaxFlt
        absMaxAllStr = absMaxStr
    else:
        absMaxAll = abs2MinFlt
        absMaxAllStr = absMinStr
        
    #make sure specified magnitude is not larger than maximum difference of two rasters
        
    apeValue2 = float(apeValue)
    
    if apeValue2 > absMaxAll:
        GP.AddError("Magnitude selected: [" + str(apeValue2) + "] is greater than maximum difference between scenarios selected: [" + absMaxAllStr + "]")
        sys.exit(0)

    negApeValue = (apeValue2 * -1)
    
    GP.AddMessage("Maximum difference between grid values: " + absMaxAllStr)
    GP.AddMessage("Magnitude of difference selected: " + str(apeValue))
    GP.AddMessage("Values above %s or below %s will be considered within the area of potential effects." % (str(apeValue), str(negApeValue)) )

    outputCon = UW.tempGrid()

    #highlight areas that are greater than value selected
    arcpy.gp.Con_sa(outputInt, "1", outputCon, "0", "value >" + str(apeValue) + " OR value <-" + str(apeValue))

    #convert grid to polygon shapefile
    arcpy.RasterToPolygon_conversion(outputCon, outShape, "NO_SIMPLIFY", "VALUE")

    
    UW.deleteGrid(outputMinus)
    UW.deleteGrid(outputCon)

    if outRast == "#":
        UW.deleteGrid(outputInt)
  
    
def main():
    
    # Get the script parameters    
    if len(sys.argv) < 5:
        GP.AddError("ape_jjr.py did not receive all five required arguments")
        sys.exit(1)
    else:
        inRast1 = sys.argv[1]
        inRast2 = sys.argv[2]
        apeValue = sys.argv[3]
        outShape = sys.argv[4]
        outRast = sys.argv[5]
   
    # ScratchWorkspace
    if GP.ScratchWorkspace == "":
        GP.AddError("A scratch workspace must be defined in your Environmental settings for this script to function properly.")
        sys.exit(0)

    if inRast1 == inRast2:
        GP.AddError("Input raster and comparison raster must be different rasters.")
        sys.exit(0)
        
    if GP.Exists(inRast1) and GP.Exists(inRast2):
        try:
            calcAPE(inRast1, inRast2, apeValue, outShape, outRast)
        except:
            GP.AddError(GP.GetMessages(2))
            raise
    else:
        GP.AddError("Invalid Grid" ) 
        sys.exit(0)

        
if __name__ == "__main__":
    main()
    
   
    
    

                  
    



    
    
        
    
            


        
